/* This program is part of a faithfully-rounded, division-free l2
   vector norm with avoidance of spurious underflow and overflow.
   
   Author: Christoph Lauter,

           Université Pierre et Marie Curie Paris 6, UPMC, LIP6,
	   PEQUAN team.

   implementing the algorithm described in the paper

   Graillat, Lauter, Tang, Yamanaka and Oishi: Efficient calculations
   of faithfully rounded l2-norms of n-vectors.

   This program is

   Copyright (C) 2014 Université Pierre et Marie Curie Paris 6, UPMC,
   LIP6, PEQUAN team.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>. 

*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include "faithfulnormmpfr.h"

/* Compile me with something like 

   gcc -std=c99 -O3 -msse3 -c faithfulnormnooverflow.c
   gcc -std=c99 -O3 -c faithfulnormmpfr.c
   gcc -std=c99 -O0 -c faithfulnormtester.c
   gcc -std=c99 -O0 -o faithfulnorm faithfulnormtester.o faithfulnormnooverflow.o faithfulnormmpfr.o -lm -lmpfr

   If you are not testing on an Intel64 based system (or if there is
   any other reason not to use Intel64 assembly), define
   NO_ASSEMBLY_TIMING by adding -DNO_ASSEMBLY_TIMING=1 to the compile
   lines.

*/

#ifdef NO_ASSEMBLY_TIMING

#define READ_TIME_COUNTER(time) do {                   \
  struct timeval t;                                    \
  gettimeofday(&t,NULL);                               \
  *time = (((uint64_t) t.tv_sec) *                     \
	   ((uint64_t) 1000000)) +                     \
    ((uint64_t) t.tv_usec);                            \
} while ( __ALWAYS_FALSE );

#else

#define READ_TIME_COUNTER(time)                        \
  __asm__ __volatile__(                                \
          "xorl %%eax,%%eax\n\t"                       \
          "cpuid\n\t"                                  \
          "rdtsc\n\t"                                  \
          "movl %%eax,(%0)\n\t"                        \
          "movl %%edx,4(%0)\n\t"                       \
          "xorl %%eax,%%eax\n\t"                       \
          "cpuid\n\t"                                  \
          : /* nothing */                              \
          : "S"((time))                                \
          : "eax", "ebx", "ecx", "edx", "memory")

#endif

/* Some symbolic constants */
#define FULL_EXPONENT_RANGE        0
#define NO_OVERFLOW                1
#define AROUND_ONE                 2
#define REALLY_SMALL               3
#define NORM_GRADUALLY_UNDERFLOWS  4
#define HALF_ULP_CASES             5
#define MONOTONE_INCREASING        6
#define MONOTONE_DECREASING        7
#define NETLIB_SPURIOUS_UNDERFLOW  8


/* Compute a correctly rounded double precision square root of op
   rounded in rounding direction rnd. rop is supposed to have at least
   53 bits of precision.
*/
int mpfr_sqrt_d(mpfr_t rop, mpfr_t op, mp_rnd_t rnd) {
  mp_exp_t oldEmin, oldEmax;
  mp_prec_t oldDefaultPrec;
  int ternaryA, ternaryB;
  mpfr_t temp;

  oldEmin = mpfr_get_emin();
  oldEmax = mpfr_get_emax();
  oldDefaultPrec = mpfr_get_default_prec();

  mpfr_set_default_prec(53);
  mpfr_set_emin(-1073); mpfr_set_emax(1024);
  
  mpfr_init(temp);

  ternaryA = mpfr_sqrt(temp, op, rnd);
  ternaryB = mpfr_subnormalize(temp, ternaryA, rnd);

  mpfr_set(rop, temp, GMP_RNDN); /* exact because prec(rop) >= 53 */

  mpfr_set_emin(oldEmin);
  mpfr_set_emax(oldEmax);
  mpfr_set_default_prec(oldDefaultPrec);

  mpfr_clear(temp);

  return ternaryB;
}

void computeUlp(mpfr_t ulp, mpfr_t yd) {
  dblcast ydb, sdb;

  ydb.d = mpfr_get_d(yd, GMP_RNDN); /* exact */

  if (ydb.i == SIGN_MASK) {
    sdb.i = ydb.i - 1;
    ydb.i = sdb.i - 2;
    ydb.d = sdb.d - ydb.d;
  } else {
    if (!((ydb.i & SIGN_MASK) == SIGN_MASK)) {
      sdb.i = ydb.i + 1;
      if (sdb.i == SIGN_MASK) {
	sdb.i = ydb.i - 1;
	ydb.i = ydb.i - 2;
      }
      ydb.d = sdb.d - ydb.d;
    }
  }

  mpfr_set_d(ulp, ydb.d, GMP_RNDN);

}

int isFaithfulNorm(double y, CONST double * RESTRICT x, int n, int displayErrors, int *underflowCases, int *overflowCases, mpfr_t eps, mpfr_t ulpErr) {
  mpfr_t acc, t, yd, yu, ym, yAcc, temp, ulp;
  int i, res;
  dblcast xdb, ydb, adb, bdb;

  mpfr_init2(acc, (2 * EXPO_MAX) - (2 * EXPO_MIN) + 1 + (2 * PRECISION));
  mpfr_init2(t, (2 * PRECISION));
  mpfr_init2(yd, PRECISION);
  mpfr_init2(yu, PRECISION);
  mpfr_init2(ym, PRECISION);
  mpfr_init2(yAcc, 500);
  mpfr_init2(temp, 500);
  mpfr_init2(ulp, 64);
  
  mpfr_set_d(ym, y, GMP_RNDN); /* exact */

  mpfr_set_si(acc, 0, GMP_RNDN); /* exact */
  for (i=0; i<n; i++) {
    mpfr_set_d(t, x[i], GMP_RNDN); /* exact */
    mpfr_mul(t, t, t, GMP_RNDN); /* exact */
    mpfr_add(acc, acc, t, GMP_RNDN); /* exact */
  }
  mpfr_sqrt_d(yd, acc, GMP_RNDD); /* one correct rounding DOWN */
  mpfr_sqrt_d(yu, acc, GMP_RNDU); /* one correct rounding UP */
  mpfr_sqrt(yAcc, acc, GMP_RNDN); /* Some error for max. eps estimation only */

  if (!mpfr_zero_p(yAcc)) {
    mpfr_sub(temp, ym, yAcc, GMP_RNDN);
    mpfr_div(temp, temp, yAcc, GMP_RNDN);
    mpfr_abs(temp, temp, GMP_RNDN);
    if (mpfr_cmp(temp, eps) > 0) {
      mpfr_set(eps, temp, GMP_RNDN);
    }
  }

  computeUlp(ulp, yd);
  mpfr_sub(temp, ym, yAcc, GMP_RNDN);
  mpfr_abs(temp, temp, GMP_RNDN);
  mpfr_div(temp, temp, ulp, GMP_RNDN);
  if (mpfr_cmp(temp, ulpErr) > 0) {
    mpfr_set(ulpErr, temp, GMP_RNDN);
  }
  
  res = (mpfr_equal_p(ym,yd) || mpfr_equal_p(ym, yu));

  if ((!res) && (displayErrors)) {
    ydb.d = y;
    adb.d = mpfr_get_d(yd, GMP_RNDN);
    bdb.d = mpfr_get_d(yu, GMP_RNDN);    
    printf("Not faithfully rounded: n = %d, y = %1.50e = 0x%016llx, should be one of 0x%016llx or 0x%016llx\n", n, y, 
	   (long long unsigned int) ydb.i, (long long unsigned int) adb.i, (long long unsigned int) bdb.i);
    printf("x = [\n");
    for (i=0; i<n; i++) {
      xdb.d = x[i];
      printf("%1.50e = 0x%016llx\n", x[i], (long long unsigned int) xdb.i);
    }
    printf("]\n");
  }

  ydb.d = y;
  if (((ydb.i & SIGN_MASK) >> (PRECISION - 1)) == 0ull) {
    (*underflowCases)++;
  }
  if ((ydb.i & SIGN_MASK) == SIGN_MASK) {
    (*overflowCases)++;
  }


  mpfr_clear(acc);
  mpfr_clear(t);
  mpfr_clear(yd);
  mpfr_clear(yu);
  mpfr_clear(yAcc);
  mpfr_clear(temp);
  mpfr_clear(ulp);

  return res;
}

double buildFPNumber(int sign, int expo, uint64_t mant) {
  dblcast xdb;
  double res;
  
  if (expo + BIAS >= 0) {
    xdb.i = (((uint64_t) (expo + BIAS)) << (PRECISION - 1)) | 
      ((mant >> (64 - PRECISION) & (~(((uint64_t) 1) << (PRECISION - 1)))));
  } else {
    xdb.i = ((mant >> (64 - PRECISION - (expo + BIAS)) & (~(((uint64_t) 1) << (PRECISION - 1 + (expo + BIAS))))));
  }
  res = (sign ? -xdb.d : xdb.d); 
  
  return res;
}

uint64_t randomUint64() {
  uint64_t res;
  int i;
  unsigned long int r;

  res = 0;
  for (i=0; i<8; i++) {
    res <<= 8;
    r = rand();
    res |= (uint64_t) (r & 0xff);
  }

  return res;
}

int randomIntInRange(int min, int max) {
  int res;

  do {
    res = min + ((int) (((double) (max - min + 1)) * (((double) rand()) / ((double) RAND_MAX))));
  } while ((res < min) || (res > max));
  
  return res;
}

double randomDoubleMOneOne() {
  uint64_t mant;
  int expo, sign;

  expo = randomIntInRange(EXPO_MIN, 0);
  sign = (rand() <= (RAND_MAX / 2));
  mant = randomUint64();
  return buildFPNumber(sign, expo, mant);
}

int log2int(int n) {
  int i, l;
  for (i=n, l = 0; i > 0; i >>= 1, l++);
  return l;
}

double sqrt(double);

void makeRandomVector(double *x, int n, int testCase) {
  int i;
  uint64_t mant;
  int expo, sign;
  int minExpo, maxExpo;
  CONST double exactSquare = 1.16415432822132222656641431512980489060282707214355e-10; /* 2^-33 + 2^-53 - 2^-12 * 2^-53 */
  CONST double perturb = 6.617444900424221398971269536559702828526496887207e-24; /* 1/2 * 2^-33 * 2^-53 * 2^10 */
  CONST double two1022 = 4.49423283715578976932326297697256183404494244735577e307; /* 2^1022 */
  CONST double two969 = 4.9896007738367995291409317825920964151686281510886e291; /* 2^969 */
  double valsq;

  if (testCase == HALF_ULP_CASES) {
    if (n > 0) {
      x[0] = 1.0;
      for (i=1; i<n; i++) {
	valsq = exactSquare + perturb * randomDoubleMOneOne();
	x[i] = sqrt(valsq);
      }
    }
  } else {
    if (testCase == MONOTONE_INCREASING) {
      for (i=0;i<n;i++) {
	x[i] = 1.0 + ((double) i) / ((double) n);
      }
    } else {
      if (testCase == MONOTONE_DECREASING) {
	for (i=0;i<n;i++) {
	  x[i] = 1.0 + ((double) (n-i)) / ((double) n);
	}
      } else {
	if (testCase == NETLIB_SPURIOUS_UNDERFLOW) {
	  if (n > 1) {
	    x[0] = two1022;
	    x[1] = two969;
	    minExpo = EXPO_MIN;
	    maxExpo = -1;  
	    for (i=2;i<n;i++) {
	      expo = randomIntInRange(minExpo, maxExpo);
	      sign = (rand() <= (RAND_MAX / 2));
	      mant = randomUint64();
	      x[i] = buildFPNumber(sign, expo, mant);
	    }
	  }
	} else {
	  switch (testCase) {
	  case NO_OVERFLOW:
	    minExpo = EXPO_MIN;
	    maxExpo = EXPO_MAX - log2int(n) - 2;  
	    break;
	  case AROUND_ONE:
	    minExpo = EXPO_MIN_ONE_RANGE;
	    maxExpo = EXPO_MAX_ONE_RANGE;
	    break;
	  case REALLY_SMALL:
	    minExpo = EXPO_MIN;
	    maxExpo = EXPO_MAX_SMALL_RANGE;
	    break;
	  case NORM_GRADUALLY_UNDERFLOWS:
	    minExpo = EXPO_MIN;
	    maxExpo = EXPO_LEAST_NORMAL - log2int(n) - 2;
	    break;
	  case FULL_EXPONENT_RANGE:
	  default:
	    minExpo = EXPO_MIN;
	    maxExpo = EXPO_MAX;  
	  }

	  for (i=0; i<n; i++) {
	    expo = randomIntInRange(minExpo, maxExpo);
	    sign = (rand() <= (RAND_MAX / 2));
	    mant = randomUint64();
	    x[i] = buildFPNumber(sign, expo, mant);
	  }
	}
      }
    }
  }
}

uint64_t timeNorm(double (*norm)(CONST double * RESTRICT, unsigned int),
                  double (*dummy)(CONST double * RESTRICT, unsigned int),
                  CONST double * RESTRICT x,
                  unsigned int n, int runs) {
  int i;
  uint64_t before, after, callOverhead, overallCallOverhead, time, overallTime;
  double res;
  volatile double sink;
  
  overallCallOverhead = 0;
  for (i=0; i<runs; i++) {
    do {
      READ_TIME_COUNTER(&before);
      res = dummy(x, n);
      READ_TIME_COUNTER(&after);
      sink = res;
    } while (before >= after);
    callOverhead = after - before;
    overallCallOverhead += callOverhead;
  }

  overallTime = 0;
  for (i=0; i<runs; i++) {
    do {
      READ_TIME_COUNTER(&before);
      res = norm(x, n);
      READ_TIME_COUNTER(&after);
      sink = res;
    } while (before >= after);
    time = after - before;
    overallTime += time;
  }

  return overallTime - overallCallOverhead;
}

int testNorm(double (*norm)(CONST double * RESTRICT, unsigned int),
             CONST double * RESTRICT x,
             unsigned int n,
	     int displayErrors,
	     int *underflowCases,
	     int *overflowCases,
	     mpfr_t eps,
	     mpfr_t ulpErr) {
  double y;

  y = norm(x, n);
  
  if (isFaithfulNorm(y, x, n, displayErrors, underflowCases, overflowCases, eps, ulpErr)) return 1;

  return 0;
}

void safeFree(void *ptr) {
  free(ptr);
}

void *safeCalloc(size_t nmemb, size_t size) {
  void *ptr;

  ptr = calloc(nmemb, size);
  if (ptr == NULL) {
    fprintf(stderr, "Could not allocate enough memory.\n");
    exit(2);
  }

  return ptr;
}

double log2(double);

int timeAndTestNorm(double (*norm)(CONST double * RESTRICT, unsigned int),
                    double (*dummy)(CONST double * RESTRICT, unsigned int),
                    char *normName,
                    unsigned int n, int k, int runs, 
		    int displayErrors, int testCase) {
  uint64_t overallTime;
  int overallFaithfulResults, underflowCases, overflowCases;
  int i;
  double *x;
  double avgTime, ratioFaithful, ratioUnderflow, ratioOverflow;
  mpfr_t eps, ulpErr;
  double epsDouble, ulpErrDouble;
  
  x = (double *) safeCalloc(n, sizeof(double));

  overallTime = 0;
  overallFaithfulResults = 0;
  underflowCases = 0;
  overflowCases = 0;
  mpfr_init2(eps, 53);
  mpfr_set_si(eps, 0, GMP_RNDN);
  mpfr_init2(ulpErr, 53);
  mpfr_set_si(ulpErr, 0, GMP_RNDN);
  for (i=0; i<k; i++) {
    makeRandomVector(x, n, testCase);
    overallTime += timeNorm(norm, dummy, x, n, runs);
    overallFaithfulResults += testNorm(norm, x, n, displayErrors, &underflowCases, &overflowCases, eps, ulpErr);
  }
  
  avgTime = ((double) overallTime) / (((double) n) * ((double) k) * ((double) runs));
  ratioFaithful = ((double) overallFaithfulResults) / ((double) k);
  ratioUnderflow = ((double) underflowCases) / ((double) k);
  ratioOverflow = ((double) overflowCases) / ((double) k);

  epsDouble = mpfr_get_d(eps, GMP_RNDN);
  ulpErrDouble = mpfr_get_d(ulpErr, GMP_RNDN);

  printf("Test and timing results for %s:\n", normName);
  printf("\tAverage timing:                    %10.6f time units per element\n", avgTime);
  printf("\tPercentage of underflowed results: %6.2f%%\n", ratioUnderflow * 100.0);
  printf("\tPercentage of overflowed results:  %6.2f%%\n", ratioOverflow * 100.0);
  printf("\tPercentage of faithful results:    %6.2f%%\n", ratioFaithful * 100.0);
  printf("\tMaximum relative error:              %1.5e = 2^(%6.2f)\n", epsDouble, log2(epsDouble));
  if (ulpErrDouble < 1000) {
    printf("\tMaximum error in ulps:               %3.4f\n", ulpErrDouble);
  } else {
    printf("\tMaximum error in ulps:               %1.5e\n", ulpErrDouble);
  }
  printf("\n");

  safeFree(x);

  mpfr_clear(eps);
  mpfr_clear(ulpErr);

  return (overallFaithfulResults == k);
}

void printUsage(char *str) {
  fprintf(stderr, "Usage: %s <vector length> <number of tests> <timing runs>\n",str);
  fprintf(stderr, "\n");
}

int readArg(int *r, char *str, int min) {
  int res;
  char *end;
  
  if (str[0] == '\0') return 0;
  res = strtol(str, &end, 10);
  if (end[0] != '\0') return 0;

  if (res < min) return 0;

  *r = res;
  return 1;
}

int main(int argc, char **argv) {
  int okay = 1;
  int k, runs;
  int sn;
  unsigned int n;

  if (argc < 4) {
    printUsage(argv[0]);
    return 2;
  }

  if (!readArg(&sn, argv[1], 1)) {
    printUsage(argv[0]);
    return 2;
  }
  n = sn;

  if (!readArg(&k, argv[2], 1)) {
    printUsage(argv[0]);
    return 2;
  }

  if (!readArg(&runs, argv[3], 1)) {
    printUsage(argv[0]);
    return 2;
  }

  printf("Testing with vectors where overflow is possible due to summing of huge values:\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, FULL_EXPONENT_RANGE);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, FULL_EXPONENT_RANGE);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, FULL_EXPONENT_RANGE);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, FULL_EXPONENT_RANGE);
  printf("\n");

  printf("Testing with vectors where overflow is NOT possible due to summing of huge values:\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, NO_OVERFLOW);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, NO_OVERFLOW);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, NO_OVERFLOW);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, NO_OVERFLOW);
  printf("\n");

  printf("Testing with vectors with values in a small exponent range around exponent 0 (value 1.0):\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, AROUND_ONE);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, AROUND_ONE);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, AROUND_ONE);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, AROUND_ONE);
  printf("\n");

  printf("Testing with vectors with very small values:\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, REALLY_SMALL);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, REALLY_SMALL);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, REALLY_SMALL);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, REALLY_SMALL);
  printf("\n");

  printf("Testing with vectors for which the norm gradually underflows:\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, NORM_GRADUALLY_UNDERFLOWS);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, NORM_GRADUALLY_UNDERFLOWS);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, NORM_GRADUALLY_UNDERFLOWS);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, NORM_GRADUALLY_UNDERFLOWS);
  printf("\n");

  printf("Testing with vectors with half-ulp entries:\n");
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, HALF_ULP_CASES);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, HALF_ULP_CASES);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, HALF_ULP_CASES);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, HALF_ULP_CASES);
  printf("\n");

  printf("Testing with vectors with monotone increasing entries:\n"); 
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, MONOTONE_INCREASING);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, MONOTONE_INCREASING);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, MONOTONE_INCREASING);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, MONOTONE_INCREASING);
  printf("\n");

  printf("Testing with vectors with monotone decreasing entries:\n"); 
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, MONOTONE_DECREASING);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, MONOTONE_DECREASING);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, MONOTONE_DECREASING);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, MONOTONE_DECREASING);
  printf("\n");

  printf("Testing with vectors with entries provoking lots of spurious underflow exceptions in netlib:\n"); 
  timeAndTestNorm(naiveNorm, dummyFunc, "naive norm", n, k, runs, 0, NETLIB_SPURIOUS_UNDERFLOW);
  timeAndTestNorm(netlibNorm, dummyFunc, "netlib norm", n, k, runs, 0, NETLIB_SPURIOUS_UNDERFLOW);
  timeAndTestNorm(faithfulNormMpfr, dummyFunc, "norm in MPFR", n, k, runs, 1, NETLIB_SPURIOUS_UNDERFLOW);
  okay = okay & !!timeAndTestNorm(faithfulNorm, dummyFunc, "faithful norm", n, k, runs, 1, NETLIB_SPURIOUS_UNDERFLOW);
  printf("\n");


  
  if (!okay) return 1;
  return 0;
}



